/*
 * @(#)EntityFigure.java  1.1.1  2006-12-05
 *
 * Copyright (c) 2006 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */

package ch.hslu.cm.rer.diagram;

import ch.hslu.cm.rer.model.RERAttribute;
import ch.hslu.cm.rer.model.Entity;
import ch.hslu.cm.rer.model.RelationalERModel;
import ch.hslu.cm.*;
import ch.hslu.cm.simulation.*;
import java.awt.*;
import java.awt.geom.*;
import java.awt.event.*;
import java.io.IOException;
import java.util.*;
import javax.swing.*;
import javax.swing.undo.*;
import org.jhotdraw.app.action.ActionUtil;
import static org.jhotdraw.draw.AttributeKeys.*;
import org.jhotdraw.draw.*;
import org.jhotdraw.draw.connector.Connector;
import org.jhotdraw.draw.event.FigureAdapter;
import org.jhotdraw.draw.event.FigureEvent;
import org.jhotdraw.draw.handle.Handle;
import org.jhotdraw.draw.handle.MoveHandle;
import org.jhotdraw.draw.layouter.VerticalLayouter;
import org.jhotdraw.geom.*;
import org.jhotdraw.undo.CompositeEdit;
import org.jhotdraw.util.ResourceBundleUtil;
import org.jhotdraw.xml.DOMInput;
import org.jhotdraw.xml.DOMOutput;
/**
 * EntityFigure.
 *
 * @author Werner Randelshofer
 * @version 1.1.1 2006-12-02 Fixed rendering of presentation figure.
 * <br>1.1 2006-02-21 Support for super-sub-type added.
 * <br>1.0 2006-01-18 Created.
 */
public class EntityFigure extends GraphicalCompositeFigure
        implements DiagramFigure, AttributedElementListener {
    private Entity model;
    /**
     * This adapter is used, to connect a TextFigure with the name of
     * the Entity model.
     */
    private static class NameAdapter extends FigureAdapter {
        private EntityFigure target;
        public NameAdapter(EntityFigure target) {
            this.target = target;
        }
    @Override
        public void attributeChanged(final FigureEvent e) {
            if (e.getAttribute().equals(TEXT)) {
                target.model.setName((String) e.getNewValue());
                
                target.fireUndoableEditHappened(new AbstractUndoableEdit() {
                    @Override
                    public String getPresentationName() {
                        ResourceBundleUtil labels = RelationalERModel.labels;
                        return labels.getString("name");
                    }
                    @Override
                    public void undo() throws CannotUndoException {
                        super.undo();
                        target.model.setName((String) e.getOldValue());
                    }
                    @Override
                    public void redo() throws CannotUndoException {
                        super.redo();
                        target.model.setName((String) e.getNewValue());
                    }
                });
            }
        }
    }
    private NameAdapter nameAdapter;
    
    private class EntityTypeAction extends AbstractAction {
        private int type;
        public EntityTypeAction(String name, int type) {
            super(name);
            this.type = type;
            putValue(ActionUtil.SELECTED_KEY, model.getType() == type);
            putValue(ActionUtil.BUTTON_GROUP_KEY, model);
        }
        @Override
        public void actionPerformed(ActionEvent event) {
            final int oldValue = model.getType();
            model.setType(type);
            fireUndoableEditHappened(new AbstractUndoableEdit() {
                @Override
                public String getPresentationName() {
                    ResourceBundleUtil labels = RelationalERModel.labels;
                    return labels.getString("entityType");
                }
                @Override
                public void undo() throws CannotUndoException {
                    super.undo();
                    model.setType(oldValue);
                }
                @Override
                public void redo() throws CannotUndoException {
                    super.redo();
                    model.setType(type);
                }
            });
        }
    };
    
    
    
    /**
     * This adapter is used, to connect a TextFigure for an attribute with
     * the Entity model.
     */
    private class AttributeAdapter extends FigureAdapter {
        private int index;
        public AttributeAdapter(int index) {
            this.index = index;
        }
        
        @Override
        public void attributeChanged(final FigureEvent e) {
            if (e.getAttribute().equals(TEXT)) {
                String text = (String) e.getNewValue();
                if (text == null || text.length() == 0) {
                    model.removeAttribute(index);
                } else {
                    model.getAttribute(index).setName(text);
                    fireUndoableEditHappened(new AbstractUndoableEdit() {
                        @Override
                        public String getPresentationName() {
                            ResourceBundleUtil labels = RelationalERModel.labels;
                            return labels.getString("attribute");
                        }
                        @Override
                        public void undo() throws CannotUndoException {
                            super.undo();
                            model.getAttribute(index).setName((String) e.getOldValue());
                        }
                        @Override
                        public void redo() throws CannotUndoException {
                            super.redo();
                            model.getAttribute(index).setName((String) e.getNewValue());
                        }
                    });
                }
            }
        }
    }
    
    /** Creates a new instance. */
    public EntityFigure() {
        this(Entity.TYPE_FUNDAMENTAL);
    }
    public EntityFigure(int entityType) {
        //super(new ShadowRectangleFigure());
        super(new RectangleFigure());
        
        setLayouter(new VerticalLayouter());
        
        RectangleFigure nameCompartmentPF = new RectangleFigure();
        nameCompartmentPF.set(STROKE_COLOR, null);
        nameCompartmentPF.set(FILL_COLOR, null);
        nameCompartmentPF.setAttributeEnabled(STROKE_COLOR, false);
        nameCompartmentPF.setAttributeEnabled(FILL_COLOR, false);
        nameCompartmentPF.setAttributeEnabled(STROKE_TYPE, false);
        ListFigure nameCompartment = new ListFigure(nameCompartmentPF);
        ListFigure attributeCompartment = new ListFigure();
        attributeCompartment.set(STROKE_COLOR, null);
        nameCompartmentPF.set(FILL_COLOR, null);
        attributeCompartment.setAttributeEnabled(STROKE_COLOR, false);
        attributeCompartment.setAttributeEnabled(FILL_COLOR, false);
        attributeCompartment.setAttributeEnabled(STROKE_TYPE, false);
        SeparatorLineFigure separator1 = new SeparatorLineFigure();
        
        separator1.setVisible(false);
        attributeCompartment.setVisible(false);
        
        Insets2D.Double insets = new Insets2D.Double(6,10,6,10);
        LAYOUT_INSETS.set(nameCompartment, insets);
        LAYOUT_INSETS.set(attributeCompartment, insets);
        
        TextFigure nameFigure;
        nameCompartment.add(nameFigure = new TextFigure());
        nameFigure.addFigureListener(nameAdapter = new NameAdapter(this));
        FONT_BOLD.set(nameFigure, true);
        
        applyAttributes(this);
        applyAttributes(getPresentationFigure());
        
        add(nameCompartment);
        add(separator1);
        add(attributeCompartment);
        
        setModel(createEntity(entityType));
        
        setAttributeEnabled(STROKE_TYPE, false);
        setAttributeEnabled(STROKE_DASHES, false);
        setAttributeEnabled(FONT_BOLD, false);
        setAttributeEnabled(FONT_ITALIC, false);
        setAttributeEnabled(FONT_UNDERLINE, false);
    }
    // DRAWING
    // SHAPE AND BOUNDS
    // ATTRIBUTES
    // EDITING
    // CONNECTING
    // COMPOSITE FIGURES
    // CLONING
    // EVENT HANDLING
    
    public Entity getSimulatedEntityType() {
        return (Entity) getModel();
    }
    
    /*package*/ Figure getAttributeFigure(RERAttribute attribute) {
        // It is possible, that connectors are informed about an attribute
        // addition or removal before we have adjusted our attribute figures.
        // Therefore we use min to avoid an index out of bounds exception.
        Entity set = getSimulatedEntityType();
        for (int i=0, n=Math.min(set.getAttributeCount(), getAttributeCompartment().getChildCount()); i < n; i++) {
            if (set.getAttribute(i) == attribute) {
                return getAttributeCompartment().getChild(i);
            }
        }
        return null;
    }
    
    /**
     * Gets a connector for this figure at the given location.
     * A figure can have different connectors at different locations.
     */
    @Override
    public Connector findConnector(Point2D.Double p, ConnectionFigure prototype) {
        // FIXME - Get rid of instanceof checks. This is not OO!
        if ((prototype instanceof RelationshipFigure) && isAttributesVisible()) {
            for (int i=0, n = getAttributeCompartment().getChildCount(); i < n; i++) {
                if (getAttributeCompartment().getChild(i).contains(p)) {
                    RERAttributeConnector c = new RERAttributeConnector(this, getModel().getAttribute(i));
                    c.updateAnchor(p);
                    return c;
                }
            }
        }
       // if (getNameCompartment().contains(p)) {
            RERAttributeConnector c = new RERAttributeConnector(this, null);
            c.updateAnchor(p);
            return c;
        //} else {
        //    return null;
       // }
    }
    
    
    @Override
    public Collection<Handle> createHandles(int detailLevel) {
        LinkedList<Handle> handles = new LinkedList<Handle>();
        if (detailLevel == 0) {
            MoveHandle.addMoveHandles(this, handles);
        }
        return handles;
    }
    
    protected Entity createEntity(int entityType) {
        Entity entity = new Entity();
        entity.setType(entityType);
        return entity;
    }
    
    private void updateAttributeCompartment() {
        getAttributeCompartment().removeAllChildren();
        for (int i=0; i < model.getAttributeCount(); i++) {
            RERAttribute attr = model.getAttribute(i);
            TextFigure f = new TextFigure(attr.getName());
            f.addFigureListener(new AttributeAdapter(i));
            f.set(CompositeFigure.LAYOUT_INSETS, new Insets2D.Double(1,0,3,0));
            applyAttributes(f);
            f.set(FONT_UNDERLINE, attr.isPrimaryKey());
            if (attr.isForeignKey()) {
                RoundRectangleFigure rrf = new RoundRectangleFigure();
                FILL_COLOR.set(rrf, null);
                STROKE_WIDTH.set(rrf, 0.5d);
                rrf.setArc(8f,8f);
                f.setDecorator(rrf);
                DECORATOR_INSETS.set(f, new Insets2D.Double(1d, 4d, 1d, 4d));
            }
            getAttributeCompartment().add(f);
        }
    }
    
    public void setModel(Entity m) {
        willChange();
        if (model != null) {
            model.removeAttributedElementListener(this);
        }
        model = m;
        if (model != null) {
            model.adddAttributedElementListener(this);
            ((TextFigure) getNameCompartment().getChild(0)).setText(model.getName());
            
            updateAttributeCompartment();
        }
        layout();
        changed();
    }
    
    @Override
    public Entity getModel() {
        return model;
    }
    
    private Diagram getDiagram() {
        return (Diagram) getDrawing();
    }
    private Simulation getSimulation() {
        return getDiagram().getSimulation();
    }
    
    @Override
    public void addNotify(Drawing drawing) {
        super.addNotify(drawing);
        if ((drawing instanceof Diagram) && getModel() != null) {
            getSimulation().add(getModel());
        }
    }
    @Override
    public void removeNotify(Drawing drawing) {
        if (getDrawing() != null && getModel() != null) {
            getSimulation().remove(getModel());
        }
        super.removeNotify(drawing);
    }
    
    
    private void applyAttributes(Figure f) {
        Map<AttributeKey,Object> attr = ((AbstractAttributedFigure) getPresentationFigure()).getAttributes();
        for (Map.Entry<AttributeKey, Object> entry : attr.entrySet()) {
            f.set(entry.getKey(), entry.getValue());
        }
    }
    
    @Override
    public Collection<Action> getActions(Point2D.Double p) {
        final ResourceBundleUtil labels = RelationalERModel.labels;
        
        LinkedList<Action> actions = new LinkedList<Action>();
        Action action;
        
        Figure compartment = findCompartment(p);
        Figure item;
        
        
        action = new EntityTypeAction(labels.getString("entityTypeFundamental"), Entity.TYPE_FUNDAMENTAL);
        action.putValue(ActionUtil.SUBMENU_KEY, labels.getString("type"));
        actions.add(action);
        
        action = new EntityTypeAction(labels.getString("entityTypeAssociative"), Entity.TYPE_ASSOCIATIVE);
        action.putValue(ActionUtil.SUBMENU_KEY, labels.getString("type"));
        actions.add(action);
        
        action = new EntityTypeAction(labels.getString("entityTypeAttributive"), Entity.TYPE_ATTRIBUTIVE);
        action.putValue(ActionUtil.SUBMENU_KEY, labels.getString("type"));
        actions.add(action);
        
        action = new EntityTypeAction(labels.getString("entityTypeSupertype"), Entity.TYPE_SUPERTYPE);
        action.putValue(ActionUtil.SUBMENU_KEY, labels.getString("type"));
        actions.add(action);
        
        action = new EntityTypeAction(labels.getString("entityTypeSubtype"), Entity.TYPE_SUBTYPE);
        action.putValue(ActionUtil.SUBMENU_KEY, labels.getString("type"));
        actions.add(action);
        
        action = new EntityTypeAction(labels.getString("entityTypeSuperSubtype"), Entity.TYPE_SUPERSUBTYPE);
        action.putValue(ActionUtil.SUBMENU_KEY, labels.getString("type"));
        actions.add(action);
        
        action = new AbstractAction(labels.getString("attributeAdd")) {
    @Override
            public void actionPerformed(ActionEvent event) {
                CompositeEdit edit = new CompositeEdit(labels.getString("attributeAdd"));
                fireUndoableEditHappened(edit);
                final int index = model.getAttributeCount();
                model.addAttribute(labels.getString("attributeDefaultName"));
                setAttributesVisible(true);
                
                fireUndoableEditHappened(new AbstractUndoableEdit() {
    @Override
                    public String getPresentationName() {
                        ResourceBundleUtil labels = RelationalERModel.labels;
                        return labels.getString("entityType");
                    }
    @Override
                    public void undo() throws CannotUndoException {
                        super.undo();
                        model.removeAttribute(index);
                    }
    @Override
                    public void redo() throws CannotUndoException {
                        super.redo();
                        model.addAttribute(labels.getString("attributeDefaultName"));
                    }
                });
                fireUndoableEditHappened(edit);
            }
        };
        // action.putValue(ActionUtil.SUBMENU_KEY, labels.getString("attribute"));
        actions.add(action);
        
        if (compartment == getAttributeCompartment()) {
            final int index = getAttributeCompartment().findChildIndex(p);
            if (index != -1) {
                RERAttribute attr = getModel().getAttribute(index);
                action = new AbstractAction(labels.getFormatted("attributeRemove", getModel().getAttribute(index).getAttributeName())) {
    @Override
                    public void actionPerformed(ActionEvent event) {
                        CompositeEdit edit = new CompositeEdit(labels.getString("remove"));
                        fireUndoableEditHappened(edit);
                        final RERAttribute removed = model.removeAttribute(index);
                        fireUndoableEditHappened(new AbstractUndoableEdit() {
    @Override
                            public String getPresentationName() {
                                ResourceBundleUtil labels = RelationalERModel.labels;
                                return labels.getString("remove");
                            }
    @Override
                            public void undo() throws CannotUndoException {
                                super.undo();
                                model.addAttribute(index, removed);
                            }
    @Override
                            public void redo() throws CannotUndoException {
                                super.redo();
                                model.removeAttribute(index);
                            }
                        });
                        fireUndoableEditHappened(edit);
                    }
                };
                actions.add(action);
                
                action = new AbstractAction(labels.getFormatted("attributePrimaryKey", getModel().getAttribute(index).getAttributeName())) {
    @Override
                    public void actionPerformed(ActionEvent event) {
                        model.getAttribute(index).setPrimaryKey(! model.getAttribute(index).isPrimaryKey());
                        fireUndoableEditHappened(new AbstractUndoableEdit() {
    @Override
                            public String getPresentationName() {
                                ResourceBundleUtil labels = RelationalERModel.labels;
                                return labels.getString("attributePrimaryKey");
                            }
    @Override
                            public void undo() throws CannotUndoException {
                                super.undo();
                                model.getAttribute(index).setPrimaryKey(! model.getAttribute(index).isPrimaryKey());
                            }
    @Override
                            public void redo() throws CannotUndoException {
                                super.redo();
                                model.getAttribute(index).setPrimaryKey(! model.getAttribute(index).isPrimaryKey());
                            }
                        });
                    }
                };
                action.putValue(ActionUtil.SUBMENU_KEY, labels.getFormatted("attributeProperties", attr.getName()));
                action.putValue(ActionUtil.SELECTED_KEY, model.getAttribute(index).isPrimaryKey());
                actions.add(action);
                
                action = new AbstractAction(labels.getFormatted("attributeForeignKey", getModel().getAttribute(index).getAttributeName())) {
    @Override
                    public void actionPerformed(ActionEvent event) {
                        model.getAttribute(index).setForeignKey(! model.getAttribute(index).isForeignKey());
                        fireUndoableEditHappened(new AbstractUndoableEdit() {
    @Override
                            public String getPresentationName() {
                                ResourceBundleUtil labels = RelationalERModel.labels;
                                return labels.getString("attributeForeignKey");
                            }
    @Override
                            public void undo() throws CannotUndoException {
                                super.undo();
                                model.getAttribute(index).setForeignKey(! model.getAttribute(index).isForeignKey());
                            }
    @Override
                            public void redo() throws CannotUndoException {
                                super.redo();
                                model.getAttribute(index).setForeignKey(! model.getAttribute(index).isForeignKey());
                            }
                        });
                    }
                };
                action.putValue(ActionUtil.SUBMENU_KEY, labels.getFormatted("attributeProperties", attr.getName()));
                action.putValue(ActionUtil.SELECTED_KEY, model.getAttribute(index).isForeignKey());
                actions.add(action);
            }
        }
        
        action = new AbstractAction(isAttributesVisible() ?
            labels.getString("hideAttributes") :
            labels.getString("showAttributes")) {
    @Override
            public void actionPerformed(ActionEvent event) {
                setAttributesVisible(! isAttributesVisible());
            }
        };
        actions.add(action);
        
        return actions;
    }
    
    protected ListFigure getNameCompartment() {
        return (ListFigure) getChild(0);
    }
    protected ListFigure getAttributeCompartment() {
        return (ListFigure) getChild(2);
    }
    protected SeparatorLineFigure getSeparator1() {
        return (SeparatorLineFigure) getChild(1);
    }
    
    
    public Figure findCompartment(Point2D.Double p) {
        if (getNameCompartment().contains(p)) return getNameCompartment();
        if (getAttributeCompartment().contains(p)) return getAttributeCompartment();
        return null;
    }
    
    @Override
    public EntityFigure clone() {
        EntityFigure that = (EntityFigure) super.clone();
        that.nameAdapter = new NameAdapter(that);
        that.getNameCompartment().getChild(0).addFigureListener(that.nameAdapter);
        that.setModel((Entity) this.model.clone());
        return that;
    }
    
    @Override
    public void attributeAdded(AttributedElementEvent event) {
        updateAttributeCompartment();
        layout();
    }
    
    @Override
    public void attributeChanged(AttributedElementEvent event) {
        /*
        ((TextFigure) getAttributeCompartment().getChild(event.getIndex()))
        .setText(model.getAttribute(event.getIndex()).getName());
         */
        updateAttributeCompartment();
        layout();
    }
    
    @Override
    public void attributeRemoved(AttributedElementEvent event) {
        updateAttributeCompartment();
        layout();
    }
    
    public void nameChanged(AttributedElementEvent event) {
        ((TextFigure) getNameCompartment().getChild(0))
        .setText(model.getName());
        layout();
    }
    public void typeChanged(AttributedElementEvent event) {
        fireAreaInvalidated(getNameCompartment().getDrawingArea());
    }
    
    public int getConnectionCount() {
        return getModel().getRelationships().size();
    }
    
    public int getConnectionIndex(DiagramFigure f) {
        return getModel().getRelationships().indexOf(f.getModel());
    }
    
    public void generalizationChanged(AttributedElementEvent event) {
    }
    
    @Override
    protected void drawPresentationFigure(Graphics2D g) {
        super.drawPresentationFigure(g);
        
        AbstractAttributedFigure pf = (AbstractAttributedFigure) getPresentationFigure();
        
        g.setColor(new Color(0x8f000000 | (0xffffff & STROKE_COLOR.get(pf).getRGB()),true));
        g.setStroke(AttributeKeys.getStroke(pf));
        Rectangle2D.Double r = getNameCompartment().getBounds();
        Insets2D.Double insets = LAYOUT_INSETS.get(getNameCompartment());
        if (insets != null) {
            r.x -= insets.left;
            r.width += insets.left + insets.right;
            r.y -= insets.top;
            r.height += insets.top + insets.bottom;
        }
        GeneralPath p = null;
        switch (model.getType()) {
            case Entity.TYPE_FUNDAMENTAL :
                // nothing to do
                break;
            case Entity.TYPE_ASSOCIATIVE :
                // nothing to do
                p = new GeneralPath();
                p.moveTo((float) (r.x), (float) (r.y + r.height / 2d));
                p.lineTo((float) (r.x + r.width / 2), (float) (r.y));
                p.lineTo((float) (r.x + r.width), (float) (r.y + r.height / 2d));
                p.lineTo((float) (r.x + r.width / 2), (float) (r.y + r.height));
                p.lineTo((float) (r.x), (float) (r.y + r.height / 2));
                break;
            case Entity.TYPE_ATTRIBUTIVE :
                // nothing to do
                p = new GeneralPath();
                p.moveTo((float) (r.x), (float) (r.y + r.height - 1f));
                p.lineTo((float) (r.x + r.width / 2), (float) (r.y));
                p.lineTo((float) (r.x + r.width), (float) (r.y + r.height));
                break;
            case Entity.TYPE_SUPERTYPE :
                // nothing to do
                p = new GeneralPath();
                p.moveTo((float) (r.x + r.width), (float) (r.y + r.height - r.height / 4d));
                p.lineTo((float) (r.x + r.width - r.width / 4d), (float) (r.y + r.height));
                break;
            case Entity.TYPE_SUBTYPE :
                // nothing to do
                p = new GeneralPath();
                p.moveTo((float) (r.x), (float) (r.y + r.height / 4d));
                p.lineTo((float) (r.x + r.width / 4d), (float) (r.y));
                break;
            case Entity.TYPE_SUPERSUBTYPE :
                // nothing to do
                p = new GeneralPath();
                p.moveTo((float) (r.x + r.width), (float) (r.y + r.height - r.height / 4d));
                p.lineTo((float) (r.x + r.width - r.width / 4d), (float) (r.y + r.height));
                p.moveTo((float) (r.x), (float) (r.y + r.height / 4d));
                p.lineTo((float) (r.x + r.width / 4d), (float) (r.y));
                break;
        }
        if (p != null) {
            g.draw(p);
        }
    }
    
    protected void drawConnectors(Graphics2D g) {
        RERAttributeConnector connector = new RERAttributeConnector(this, null);
        connector.draw(g);
        for (RERAttribute attr : model.getAttributes()) {
            connector.setAttribute(attr);
            connector.draw(g);
        }
    }
    
    @Override
    public Connector findCompatibleConnector(Connector c, boolean isStart) {
        if (c.getOwner() == this) return c;
        
        // XXX - Add support for attribute connector, or copy and paste won't work!
        if (c instanceof RERAttributeConnector) {
            RERAttributeConnector ac = (RERAttributeConnector) c;
            if (ac.getAttribute() == null) {
                return new RERAttributeConnector(this, null);
            }
            Entity set = getSimulatedEntityType();
            for (int i=0, n = set.getAttributeCount(); i < n; i++) {
                RERAttribute a = set.getAttribute(i);
                if (a.getName().equals(ac.getAttribute().getName())) {
                    return new RERAttributeConnector(this, a);
                }
            }
        }
        return new RERAttributeConnector(this, null);
    }
    
    @Override
    public void read(DOMInput in) throws IOException {
        double x = in.getAttribute("x", 0d);
        double y = in.getAttribute("y", 0d);
        double w = in.getAttribute("w", 0d);
        double h = in.getAttribute("h", 0d);
        boolean b = in.getAttribute("attributesVisible", false);
        setAttributesVisible(b);
        setBounds(new Point2D.Double(x,y), new Point2D.Double(x+w,y+h));
        readAttributes(in);
        in.openElement((in.getElementCount("model") == 1) ? "model" : "Model");
        setModel((Entity) in.readObject(0));
        in.closeElement();
    }
    @Override
    public void write(DOMOutput out) throws IOException {
        Rectangle2D.Double r = getBounds();
        out.addAttribute("x", r.x);
        out.addAttribute("y", r.y);
        out.addAttribute("attributesVisible", isAttributesVisible());
        writeAttributes(out);
        out.openElement("Model");
        out.writeObject(getModel());
        out.closeElement();
    }
    @Override
    public int getLayer() {
        return RelationalERDiagram.OBJECT_LAYER;
    }
    public boolean isAttributesVisible() {
        return getAttributeCompartment().isVisible();
    }
    public void setAttributesVisible(final boolean newValue) {
        if (newValue != getAttributeCompartment().isVisible()) {
            willChange();
            basicSetAttributesVisible(newValue);
            fireUndoableEditHappened(new AbstractUndoableEdit() {
                @Override
                public String getPresentationName() {
                    String retValue;
                    
                    retValue = RelationalERModel.labels.getString("visibility");
                    return retValue;
                }
                
                @Override
                public void undo() throws CannotUndoException {
                    super.undo();
                    willChange();
                    basicSetAttributesVisible(! newValue);
                    changed();
                }
                
                @Override
                public void redo() throws CannotRedoException {
                    super.redo();
                    willChange();
                    basicSetAttributesVisible(newValue);
                    changed();
                }
                
            });
            changed();
        }
    }
    public void basicSetAttributesVisible(boolean newValue) {
        // System.out.println(this+".basicSetAttributesVisible attr="+getAttributeCompartment()+" child2="+getChild(2));
        getSeparator1().setVisible(newValue);
        getAttributeCompartment().setVisible(newValue);
    }
}
